<?php


namespace Mainto\DDDCore\Relation;


use Mainto\DDDCore\Exception\UnknownRelationException;
use Mainto\DDDCore\Relation\HasOne;
use Mainto\DDDCore\Exception\EntityKeyIsInvalid;
use Iterator;
use Mainto\DDDCore\Interfaces\ValueId;

/**
 * Class EntityArray
 * @package Mainto\DDDCore
 * @template T
 */
class EntityArray implements Iterator {
    private array $loaded = [];

    /**
     * @var $entities array<T|EntityRelation>
     */
    private array $entities;

    /**
     * EntityArray constructor.
     * @param array<T> $entities
     */
    public function __construct (array $entities) {
        $this->entities = $entities;
    }

    /**
     * @param string $relation
     * @param HasOne|HasMany $relationship
     * @param bool $hasOne
     */
    private function resolveHasManyOrOne(string $relation, $relationship, bool $hasOne = false) {
        $entityMap = $keys = [];
        foreach ($this->entities as $entity) {
            $keys[] = $key = $entity->{$relationship->getLocalKey()};
            if ($key instanceof ValueId) {
                $entityMap[$key->id()] = $entity;
            } elseif (is_string($key) || is_int($key)) {
                $entityMap[$key] = $entity;
            } else {
                throw new EntityKeyIsInvalid();
            }

            $entity->setRelation($relation, $hasOne ? null : []);
        }

        $targetEntities = $relationship->entitiesResult($keys);

        $targetEntityGroup = [];
        foreach ($targetEntities as $targetEntity) {
            $relationKey = $targetEntity->{$relationship->getTargetKey()};
            if ($relationKey instanceof ValueId) {
                $serializeValue = $relationKey->id();
            } elseif (is_string($relationKey) || is_int($relationKey)) {
                $serializeValue = $relationKey;
            } else {
                throw new EntityKeyIsInvalid();
            }

            $targetEntityGroup[$serializeValue][] = $targetEntity;
        }

        foreach ($targetEntityGroup as $serializeValue => $relationEntities) {
            if (array_key_exists($serializeValue, $entityMap)) {
                $entity = $entityMap[$serializeValue];
                if ($hasOne) {
                    $entity->setRelation($relation, array_first($relationEntities));
                } else {
                    $entity->setRelation($relation, $relationEntities);
                }
            }
        }

        $this->loaded[$relation] = true;
    }

    public function load ($relation): self {
        if (!array_key_exists($relation, $this->loaded)) {
            $first = array_first($this->entities);
            if (method_exists($first, $relation)) {
                $relationship = $first->{$relation}();
                if ($relationship instanceof HasMany) {
                    $this->resolveHasManyOrOne($relation, $relationship);
                } elseif ($relationship instanceof HasOne) {
                    $this->resolveHasManyOrOne($relation, $relationship, true);
                } else {
                    throw new UnknownRelationException();
                }
            } else {
                throw new UnknownRelationException();
            }
        }

        return $this;
    }

    /**
     * @param $relation
     * @return EntityArray
     */
    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->entities)) {
            foreach ($this->entities as $item) {
                $item->refresh($relation);
            }

            unset($this->entities[$relation]);
        }

        return $this;
    }

    /**
     * @return T
     */
    public function current () {
        return current($this->entities);
    }

    public function next () {
        next($this->entities);
    }

    public function key () {
        return key($this->entities);
    }

    public function valid (): bool {
        return key($this->entities) !== null;
    }

    public function rewind () {
        reset($this->entities);
    }

    public function filter (callable $callback): array {
        return array_filter($this->entities, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->entities as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function each(callable $callback) {
        foreach ($this->entities as $key => $value) {
            if ($callback($value, $key) === false) {
                break;
            }
        }
    }
}
