<?php


namespace Mainto\DDDCore\Relation;


use Mainto\DDDCore\Exception\EntityKeyIsInvalid;
use Iterator;
use Mainto\DDDCore\Interfaces\ValueId;

/**
 * Class EntityArray
 * @package Mainto\DDDCore
 * @template T
 */
class EntityArray implements Iterator {
    private array $loaded = [];

    /**
     * @var $entities array<T|EntityRelation>
     */
    private array $entities;

    /**
     * EntityArray constructor.
     * @param array<T> $entities
     */
    public function __construct (array $entities) {
        $this->entities = $entities;
    }

    public function load ($relation) {
        if (!array_key_exists($relation, $this->loaded)) {
            $first = array_first($this->entities);
            if (method_exists($first, $relation)) {
                $entityRelation = $first->{$relation}();
                if ($entityRelation instanceof HasMany) {
                    $entityMap = [];
                    $keys = [];
                    foreach ($this->entities as $entity) {
                        $keys[] = $key = $entity->{$entityRelation->getLocalKey()};
                        if ($key instanceof ValueId) {
                            $entityMap[$key->id()] = $entity;
                        } elseif (is_string($key) || is_int($key)) {
                            $entityMap[$key] = $entity;
                        } else {
                            throw new EntityKeyIsInvalid();
                        }

                        $entity->setRelation($relation, []);
                    }

                    $relationResult = app($entityRelation->getTargetCall()[0])->{$entityRelation->getTargetCall()[1]}($keys);

                    $relationEntityGroup = [];
                    foreach ($relationResult as $relationEntity) {
                        $relationKey = $relationEntity->{$entityRelation->getTargetKey()};
                        if ($relationKey instanceof ValueId) {
                            $serializeValue = $relationKey->id();
                        } elseif (is_string($relationKey) || is_int($relationKey)) {
                            $serializeValue = $relationKey;
                        } else {
                            throw new EntityKeyIsInvalid();
                        }

                        $relationEntityGroup[$serializeValue][] = $relationEntity;
                    }

                    foreach ($relationEntityGroup as $serializeValue => $relationEntities) {
                        if (array_key_exists($serializeValue, $entityMap)) {
                            $entity = $entityMap[$serializeValue];
                            $entity->setRelation($relation, $relationEntities);
                        }
                    }

                    $this->loaded[$relation] = true;
                }
            }
        }
    }

    /**
     * @param $relation
     */
    public function refresh ($relation) {
        if (array_key_exists($relation, $this->entities)) {
            foreach ($this->entities as $item) {
                $item->refresh($relation);
            }

            unset($this->entities[$relation]);
        }
    }

    /**
     * @return T
     */
    public function current () {
        return current($this->entities);
    }

    public function next () {
        next($this->entities);
    }

    public function key () {
        return key($this->entities);
    }

    public function valid (): bool {
        return key($this->entities) !== null;
    }

    public function rewind () {
        reset($this->entities);
    }

    public function filter (callable $callback): array {
        return array_filter($this->entities, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->entities as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function each(callable $callback) {
        foreach ($this->entities as $key => $value) {
            if ($callback($value, $key) === false) {
                break;
            }
        }
    }
}
